// Copyright 2020 Espressif Systems (Shanghai) Co. Ltd.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
#include <stdio.h>
#include <string.h>
#include "unity.h"
#include "test_utils.h"
#include "esp_log.h"
#include "screen_driver.h"
#include "ssd1306.h"
#include "ssd1307.h"
#include "ssd1322.h"

static const char *TAG = "mono lcd test";

static const unsigned char bmp_image_128_64[];
static const unsigned char bmp_image_52_24[];
static const unsigned char bmp_image_256_64_4bit[];

static void contrast_test(esp_err_t (*set_contrast)(uint8_t v))
{
    set_contrast(2);

    uint8_t v = 0, dir = 0, times = 2;
    while (1) {
        if (0 == dir) {
            if (++v >= 25) {
                dir = 1;
                if (0 == times--) {
                    break;
                }
            }
        } else {
            if (--v < 1) {
                dir = 0;
            }
        }
        ESP_LOGI(TAG, "contrast = %d", v * 10);
        set_contrast(v * 10);
        vTaskDelay(100 / portTICK_PERIOD_MS);
    }
}

TEST_CASE("Screen SSD1306 I2C test", "[screen][iot]")
{
    scr_driver_t lcd;
    scr_info_t lcd_info;
    i2c_config_t i2c_conf = {
        .mode = I2C_MODE_MASTER,
        .sda_io_num = 12,
        .sda_pullup_en = GPIO_PULLUP_ENABLE,
        .scl_io_num = 32,
        .scl_pullup_en = GPIO_PULLUP_ENABLE,
        .master.clk_speed = 100000,
    };
    i2c_bus_handle_t i2c_bus = i2c_bus_create(I2C_NUM_0, &i2c_conf);
    TEST_ASSERT_NOT_NULL(i2c_bus);

    scr_interface_i2c_config_t iface_cfg = {
        .i2c_bus = i2c_bus,
        .clk_speed = 100000,
        .slave_addr = 0x3C,
    };

    scr_interface_driver_t *iface_drv;
    TEST_ASSERT(ESP_OK == scr_interface_create(SCREEN_IFACE_I2C, &iface_cfg, &iface_drv));
    scr_controller_config_t lcd_cfg = {0};
    lcd_cfg.interface_drv = iface_drv,
    lcd_cfg.pin_num_rst = 0,
    lcd_cfg.pin_num_bckl = -1,
    lcd_cfg.rst_active_level = 0,
    lcd_cfg.bckl_active_level = 1,
    lcd_cfg.width = 128;
    lcd_cfg.height = 64;
    lcd_cfg.rotate = SCR_DIR_LRTB;
    TEST_ASSERT(ESP_OK == scr_find_driver(SCREEN_CONTROLLER_SSD1306, &lcd));
    TEST_ASSERT(ESP_OK == lcd.init(&lcd_cfg));


    TEST_ASSERT(ESP_OK == lcd.get_info(&lcd_info));
    ESP_LOGI(TAG, "Screen name:%s | width:%d | height:%d", lcd_info.name, lcd_info.width, lcd_info.height);
    TEST_ASSERT(ESP_OK == lcd.draw_bitmap(0, 0, 128, 64, (uint16_t *)bmp_image_128_64));
    vTaskDelay(1000 / portTICK_PERIOD_MS);
    uint8_t i = 0, j = 0;
    for (i = 0; i < 128 - 52; i++) {
        lcd.draw_bitmap(i, j, 52, 24, (uint16_t *)bmp_image_52_24);
        vTaskDelay(100 / portTICK_PERIOD_MS);
    }

    contrast_test(lcd_ssd1306_set_contrast);

    lcd.deinit();
    scr_interface_delete(iface_drv);
    i2c_bus_delete(&i2c_bus);
}

TEST_CASE("Screen SSD1307 I2C test", "[screen][iot]")
{
    scr_driver_t lcd;
    scr_info_t lcd_info;
    i2c_config_t i2c_conf = {
        .mode = I2C_MODE_MASTER,
        .sda_io_num = 6,
        .sda_pullup_en = GPIO_PULLUP_ENABLE,
        .scl_io_num = 7,
        .scl_pullup_en = GPIO_PULLUP_ENABLE,
        .master.clk_speed = 100000,
    };
    i2c_bus_handle_t i2c_bus = i2c_bus_create(I2C_NUM_0, &i2c_conf);
    TEST_ASSERT_NOT_NULL(i2c_bus);

    scr_interface_i2c_config_t iface_cfg = {
        .i2c_bus = i2c_bus,
        .clk_speed = 100000,
        .slave_addr = 0x3D,
    };

    scr_interface_driver_t *iface_drv;
    TEST_ASSERT(ESP_OK == scr_interface_create(SCREEN_IFACE_I2C, &iface_cfg, &iface_drv));
    scr_controller_config_t lcd_cfg = {0};
    lcd_cfg.interface_drv = iface_drv,
    lcd_cfg.pin_num_rst = 0,
    lcd_cfg.pin_num_bckl = -1,
    lcd_cfg.rst_active_level = 0,
    lcd_cfg.bckl_active_level = 1,
    lcd_cfg.width = 128;
    lcd_cfg.height = 32;
    lcd_cfg.rotate = SCR_DIR_LRTB;
    TEST_ASSERT(ESP_OK == scr_find_driver(SCREEN_CONTROLLER_SSD1307, &lcd));
    TEST_ASSERT(ESP_OK == lcd.init(&lcd_cfg));

    TEST_ASSERT(ESP_OK == lcd.get_info(&lcd_info));
    ESP_LOGI(TAG, "Screen name:%s | width:%d | height:%d", lcd_info.name, lcd_info.width, lcd_info.height);

    uint8_t i = 0, j = 0;
    for (i = 0; i < 128 - 52; i++) {
        TEST_ASSERT(ESP_OK == lcd.draw_bitmap(i, j, 52, 24, (uint16_t *)bmp_image_52_24));
        vTaskDelay(100 / portTICK_PERIOD_MS);
    }

    contrast_test(lcd_ssd1307_set_contrast);

    lcd.deinit();
    scr_interface_delete(iface_drv);
    i2c_bus_delete(&i2c_bus);
}

TEST_CASE("Screen SSD1322 SPI test", "[screen][iot]")
{
    scr_driver_t lcd;
    scr_info_t lcd_info;
    spi_config_t spi_cfg = {
        .miso_io_num = -1,
        .mosi_io_num = 37,
        .sclk_io_num = 38,
        .max_transfer_sz = 320 * 480,
    };
    spi_bus_handle_t spi_bus = spi_bus_create(2, &spi_cfg);
    TEST_ASSERT_NOT_NULL(spi_bus);

    scr_interface_spi_config_t spi_lcd_cfg = {
        .spi_bus = spi_bus,
        .pin_num_cs = 41,
        .pin_num_dc = 35,
        .clk_freq = 5000000,
        .swap_data = false,
    };

    scr_interface_driver_t *iface_drv;
    TEST_ASSERT(ESP_OK == scr_interface_create(SCREEN_IFACE_SPI, &spi_lcd_cfg, &iface_drv));

    scr_controller_config_t lcd_cfg = {0};
    lcd_cfg.interface_drv = iface_drv,
    lcd_cfg.pin_num_rst = 0,
    lcd_cfg.pin_num_bckl = -1,
    lcd_cfg.rst_active_level = 0,
    lcd_cfg.bckl_active_level = 1,
    lcd_cfg.width = 256;
    lcd_cfg.height = 64;
    lcd_cfg.rotate = SCR_DIR_LRTB;
    TEST_ASSERT(ESP_OK == scr_find_driver(SCREEN_CONTROLLER_SSD1322, &lcd));
    TEST_ASSERT(ESP_OK == lcd.init(&lcd_cfg));

    TEST_ASSERT(ESP_OK == lcd.get_info(&lcd_info));
    ESP_LOGI(TAG, "Screen name:%s | width:%d | height:%d", lcd_info.name, lcd_info.width, lcd_info.height);

    lcd.draw_bitmap(0, 0, 256, 64, (uint16_t *)bmp_image_256_64_4bit);
    vTaskDelay(1000 / portTICK_PERIOD_MS);

    uint8_t *pixels = heap_caps_malloc(lcd_info.width * lcd_info.bpp / 8 * sizeof(uint8_t), MALLOC_CAP_8BIT | MALLOC_CAP_INTERNAL);
    TEST_ASSERT_NOT_NULL(pixels);
    for (size_t i = 0; i < lcd_info.width * lcd_info.bpp / 8; i++) {
        uint32_t c = (i * (1 << lcd_info.bpp)) / (lcd_info.width * lcd_info.bpp / 8);
        pixels[i] = c | c << 4;
    }
    for (size_t i = 0; i < lcd_info.height; i++) {
        lcd.draw_bitmap(0, i, lcd_info.width, 1, (uint16_t *)pixels);
        vTaskDelay(100 / portTICK_PERIOD_MS);
    }

    for (size_t i = 0; i < lcd_info.width * lcd_info.bpp / 8; i++) {
        uint32_t c = (i * (1 << lcd_info.bpp)) / (lcd_info.width * lcd_info.bpp / 8);
        pixels[(lcd_info.width * lcd_info.bpp / 8) - i - 1] = c | c << 4;
    }
    for (size_t i = 0; i < lcd_info.height; i++) {
        lcd.draw_bitmap(0, i, lcd_info.width, 1, (uint16_t *)pixels);
        vTaskDelay(100 / portTICK_PERIOD_MS);
    }

    contrast_test(lcd_ssd1322_set_contrast);

    heap_caps_free(pixels);

    lcd.deinit();
    scr_interface_delete(iface_drv);
    spi_bus_delete(&spi_bus);
}

static const unsigned char bmp_image_128_64[1024] = {
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0XF8,
    0XFE, 0XFE, 0X3E, 0X78, 0X70, 0X70, 0X78, 0X3E, 0XFE, 0XF8, 0XF8, 0X00, 0XF8, 0XF8, 0X00, 0X00,
    0XF8, 0XF8, 0X00, 0X00, 0X00, 0X00, 0XF8, 0XFE, 0XFE, 0X0E, 0X0E, 0X0E, 0X00, 0X00, 0X00, 0XB0,
    0XB0, 0XF0, 0XF0, 0XC0, 0X00, 0XF0, 0XF0, 0XF0, 0X40, 0X70, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X0E, 0X31, 0X46, 0X31, 0X31, 0X0E,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X38, 0X46, 0X46, 0X86, 0X08, 0X86, 0X46, 0X46, 0X38, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X1F,
    0X1F, 0X1F, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X1F, 0X1F, 0X1F, 0X00, 0X43, 0X5F, 0X5C, 0X5C,
    0X7F, 0X3F, 0X00, 0X00, 0X00, 0X00, 0X03, 0X1F, 0X1F, 0X1F, 0X1F, 0X1F, 0X00, 0X00, 0X03, 0X1F,
    0X1C, 0X1F, 0X1F, 0X1F, 0X00, 0X1F, 0X1F, 0X1F, 0X00, 0X00, 0X00, 0X00, 0X00, 0X1F, 0X1F, 0X00,
    0X00, 0X1F, 0X1F, 0X00, 0X1F, 0X1F, 0X1F, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0XC0, 0X20, 0X20, 0X1C, 0X03, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X80, 0X00, 0X80, 0X00, 0X00, 0X00, 0X03, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0XE0,
    0X18, 0X18, 0X18, 0X20, 0X18, 0X18, 0X18, 0XE0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X38, 0XC4, 0XC4, 0X18, 0XC4, 0X38, 0X38, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0XC0, 0XC0, 0XC0, 0XE0, 0X20, 0X20, 0XF8, 0XF8, 0XF8, 0XD8, 0XF8, 0XF8, 0XF8, 0X3C, 0X3C, 0X3C,
    0X3C, 0X3C, 0X3C, 0X3C, 0X3C, 0X3C, 0X3C, 0XFC, 0X3C, 0X3C, 0X3C, 0XDC, 0XDC, 0XD8, 0XD8, 0XD8,
    0X38, 0X3C, 0X3C, 0X3C, 0XE4, 0XE2, 0XE1, 0XC0, 0XC0, 0XC0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X02, 0X02, 0X05, 0X1A, 0X05, 0X02, 0X02, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X01, 0X01, 0X02, 0X0C, 0X02, 0X02, 0X01, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X81, 0XE0, 0XF0, 0XF0, 0X70, 0X1C, 0X0C, 0X7E, 0X7E, 0XF3, 0XFF,
    0XFD, 0X1E, 0X1E, 0X0E, 0X01, 0X81, 0X8D, 0X8D, 0X0C, 0X0C, 0X0C, 0X0C, 0X0C, 0X0C, 0X0C, 0X0C,
    0X0C, 0X0C, 0X0C, 0X0C, 0X00, 0X00, 0X80, 0XFF, 0XFE, 0XFE, 0XFE, 0XFC, 0XE0, 0X80, 0X80, 0X10,
    0X11, 0X01, 0X83, 0X83, 0X82, 0X8C, 0X9C, 0XF1, 0XF1, 0XE1, 0X83, 0X0F, 0X0F, 0X0E, 0X1E, 0X7C,
    0X7C, 0X7C, 0XF0, 0XE0, 0X80, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X0E, 0X11, 0X11,
    0X62, 0X11, 0X0E, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X80, 0X80, 0XC0, 0XC0, 0XC0, 0X70, 0X70, 0X78, 0X78, 0X78, 0X78, 0X38, 0X78, 0X78,
    0X4E, 0X4E, 0X4E, 0X0E, 0X0E, 0X8F, 0X87, 0XB9, 0XB9, 0XBE, 0X3E, 0X3E, 0X3E, 0X3E, 0X3F, 0X3F,
    0X3F, 0X3E, 0X3E, 0X0E, 0X0E, 0X0F, 0X0F, 0X0F, 0X0E, 0X0E, 0X0E, 0X0E, 0X0E, 0X0E, 0X0E, 0X0E,
    0X0E, 0X0E, 0X0E, 0X0E, 0X8E, 0X8E, 0XCF, 0X8F, 0X8F, 0X8F, 0X8F, 0X0F, 0X0F, 0X0F, 0X0F, 0X0E,
    0X0E, 0X0E, 0X0F, 0X0F, 0X0F, 0X0F, 0X0F, 0X8F, 0X8F, 0X87, 0X87, 0X46, 0X46, 0X40, 0X00, 0X30,
    0X30, 0X30, 0X31, 0X37, 0X3F, 0X3E, 0X3E, 0X3E, 0X38, 0XF8, 0XC0, 0XC0, 0XC0, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X80, 0XE0, 0XF8,
    0X64, 0X64, 0X67, 0X1F, 0X03, 0X03, 0X00, 0X80, 0XC0, 0XE0, 0XE0, 0XE0, 0XE0, 0XF8, 0XF8, 0XF8,
    0XE0, 0XE0, 0XE0, 0XC0, 0XC0, 0X80, 0X23, 0X43, 0X43, 0X84, 0X1F, 0X78, 0XC0, 0XC0, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X03, 0X03, 0X03, 0X03, 0X03, 0X03, 0XFF, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0XC0, 0XF8, 0XF8, 0X78, 0X1C, 0XC7, 0X63, 0X63, 0XF8, 0XF8, 0XF8, 0XF8, 0XE0, 0XE0, 0XF8,
    0XF8, 0XF8, 0XF8, 0XF8, 0XF8, 0XE0, 0XE0, 0XE0, 0XC0, 0X80, 0X83, 0X83, 0X07, 0X1F, 0XBC, 0XBC,
    0XF8, 0XE0, 0XC0, 0X80, 0X80, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3E, 0X3E, 0XFF, 0XFB, 0XFF,
    0X3E, 0X3E, 0X06, 0X02, 0XFC, 0XFC, 0XFE, 0XFF, 0XFF, 0XFF, 0XFF, 0XFB, 0X03, 0X07, 0X07, 0X07,
    0X03, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFE, 0XFC, 0XFC, 0X01, 0X00, 0X00, 0XFF, 0XFF, 0XC0, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0XFF, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0XC7, 0X3F, 0X3F, 0X3C, 0X06, 0X03, 0XFC, 0XFC, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0X03,
    0X07, 0X07, 0X07, 0X03, 0X3F, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFE, 0X06, 0XFF, 0XFF,
    0XFF, 0XFF, 0X3F, 0X07, 0X07, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X01, 0X03, 0X8F,
    0X8E, 0X8E, 0XE0, 0XE0, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFE, 0XFE, 0X8E, 0X82, 0XF1, 0XF1, 0XF1,
    0X80, 0XFC, 0XFE, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFE, 0XF0, 0XF0, 0XF1, 0XF1, 0XF1, 0XF0,
    0XF0, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2, 0XF2,
    0XF2, 0XF3, 0XF3, 0XF3, 0XF3, 0XF3, 0XF3, 0XF3, 0XF3, 0XF3, 0XF1, 0XF0, 0XF0, 0XF1, 0XF1, 0XF1,
    0XF1, 0XF1, 0XF0, 0XF0, 0XF0, 0XF0, 0XFE, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFE, 0XFE, 0XE1,
    0XF1, 0XF1, 0XF1, 0XE0, 0XEC, 0XFE, 0XFE, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XF3, 0XF2, 0XE3, 0XE3,
    0XE3, 0XE1, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
};


static const unsigned char bmp_image_52_24[156] = { /* 0X02,0X01,0X34,0X00,0X18,0X00, */
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X1E, 0X82, 0X86, 0X02, 0X1E, 0X06, 0X38, 0X1E, 0X80, 0X06,
    0X1B, 0X1B, 0X00, 0X1A, 0X1C, 0X00, 0X9C, 0X80, 0X80, 0X98, 0X98, 0X80, 0X98, 0X80, 0X83, 0X81,
    0X80, 0X80, 0X00, 0X20, 0X08, 0X00, 0X00, 0X40, 0XC0, 0X04, 0X00, 0X04, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X01, 0X80, 0XC0, 0X41, 0X60, 0X60,
    0X20, 0X70, 0X31, 0X18, 0XAC, 0X24, 0X3E, 0X3E, 0X27, 0X31, 0X31, 0X25, 0X35, 0X34, 0X34, 0X20,
    0XF0, 0XBF, 0X3C, 0X39, 0X25, 0X32, 0X39, 0XBD, 0X1A, 0X42, 0X04, 0X2C, 0X38, 0X30, 0X20, 0XC0,
    0X80, 0X02, 0X06, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3C, 0X3F, 0X87,
    0XF8, 0XFC, 0XFE, 0XA6, 0X87, 0XDE, 0XFE, 0XFC, 0XE0, 0XC1, 0XDE, 0XC0, 0XE0, 0XE0, 0XE0, 0XE0,
    0XE0, 0XE0, 0XA0, 0XA0, 0XA0, 0XA0, 0XC0, 0XC0, 0XC0, 0XDE, 0XC9, 0XE2, 0XFF, 0XFF, 0XDE, 0X87,
    0X87, 0XFE, 0XFE, 0XFC, 0XAD, 0XBF, 0X1E, 0X00, 0X00, 0X00, 0X00, 0X00
};

static const unsigned char bmp_image_256_64_4bit[8192] = { /* 0X00,0X04,0X00,0X01,0X40,0X00, */
    0X00, 0X00, 0X00, 0X00, 0X00, 0X2C, 0XA0, 0X00, 0X0A, 0XCC, 0XC8, 0X53, 0X02, 0XFC, 0XCC, 0XCC,
    0XAC, 0XAA, 0XC8, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X28, 0X60, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X8A, 0X20, 0X0C, 0XAA, 0XAC, 0XAC, 0X3C, 0XCC, 0XCC, 0XCA,
    0XAA, 0X88, 0X8A, 0X80, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X38, 0XAA, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X5A, 0XCA, 0XAA, 0XA8, 0XAC, 0XCC, 0XCC, 0XCA, 0XA8,
    0X66, 0X66, 0X85, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X08, 0X8A, 0XC0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X06, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X60, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3A, 0X8A, 0X88, 0X88, 0XCC, 0XCC, 0XAA, 0XA6, 0X66,
    0X56, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X03, 0XA8, 0XAC, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3A, 0X20, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X82, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3A, 0X88, 0X88, 0X68, 0XCA, 0XAA, 0X88, 0X65, 0X58,
    0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X6A, 0X8A, 0XCA, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X36, 0XA0, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X68, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X03, 0X68, 0XA0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X20, 0X00,
    0X00, 0X28, 0X30, 0X00, 0X00, 0X00, 0X00, 0X06, 0X88, 0X66, 0X55, 0XAA, 0XA8, 0X86, 0X35, 0X80,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X08, 0XAA, 0XAC, 0XF2, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X55, 0X88, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X68, 0X30, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X03, 0X88, 0X8A, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X56, 0X53, 0X20, 0X00,
    0X00, 0XAC, 0X80, 0X00, 0X00, 0X00, 0X00, 0X00, 0X38, 0X65, 0X33, 0X38, 0X86, 0X53, 0X36, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X6A, 0XAA, 0XCC, 0XC0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X35, 0X5A, 0XA0,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X65, 0XA0, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X58, 0X88, 0XAA, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X25, 0X88, 0X86, 0X53, 0X20, 0X00,
    0X00, 0X3A, 0X82, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3A, 0XCA, 0X88, 0X88, 0X53, 0X35, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X22, 0X00, 0X3A, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X02, 0XAA, 0XAC, 0XCF, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X68, 0XAA, 0X88,
    0X52, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X26, 0X58, 0XAA,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X65, 0X8A, 0X00, 0X00, 0X00,
    0X00, 0X05, 0X68, 0X8A, 0XC3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X06, 0XAA, 0X88, 0X88, 0X88, 0X82, 0X00,
    0X00, 0X02, 0X88, 0X63, 0X20, 0X00, 0X00, 0X33, 0X33, 0X3C, 0XCA, 0XCC, 0XAC, 0XCC, 0XA8, 0X58,
    0X02, 0X8C, 0XCC, 0XCC, 0XCC, 0XA8, 0X30, 0X05, 0X3F, 0XC6, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X03, 0X6A, 0XA8, 0X65, 0X32, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X08, 0XAA, 0XAC, 0XCF, 0X00, 0X00, 0X20, 0X00, 0X00, 0X00, 0X38, 0X88, 0X88, 0X88, 0X8A,
    0XAC, 0XA6, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X05, 0X63, 0X8A,
    0XC3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X58, 0X5A, 0X80, 0X00, 0X00,
    0X00, 0X06, 0X86, 0XAA, 0XC0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X8A, 0XAA, 0XA8, 0XAC, 0XCA, 0X83, 0X00,
    0X00, 0X00, 0X00, 0X23, 0X33, 0X33, 0X3A, 0X88, 0X63, 0X33, 0X33, 0X88, 0XAC, 0XCC, 0XFF, 0XFF,
    0XA8, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0X58, 0XFF, 0XCC, 0XCA, 0X83, 0X00, 0X00, 0X06,
    0XCC, 0XA8, 0XCC, 0XCC, 0X63, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X2A, 0XAA, 0XCC, 0XFC, 0X00, 0X02, 0X36, 0X32, 0X00, 0X38, 0X65, 0X55, 0X55, 0X55, 0X55,
    0X55, 0X8A, 0XCA, 0X60, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X66, 0X38,
    0XAC, 0XA0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X28, 0X58, 0XC6, 0X00, 0X00,
    0X00, 0X36, 0X86, 0XAC, 0XC0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X2A, 0XAA, 0XCA, 0XAC, 0XCC, 0XCA, 0X83, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X8A, 0X88, 0X86, 0X55, 0X36, 0X88, 0X66, 0X88, 0XCF, 0XFF,
    0XF3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0XA6, 0X35, 0X68, 0X8A, 0XCC, 0XA5, 0X00, 0XAC,
    0XAA, 0X8F, 0XFC, 0X50, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X5A, 0XAA, 0XCC, 0XFA, 0X00, 0X03, 0X58, 0X88, 0X88, 0X30, 0X00, 0X00, 0X00, 0X02, 0X23,
    0X55, 0X65, 0X58, 0XAC, 0XA3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X36, 0X85,
    0X6A, 0XCC, 0X80, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0X85, 0XAC, 0X80, 0X00,
    0X00, 0X36, 0X86, 0XAC, 0XC0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0XAC, 0XAC, 0XCC, 0XCC, 0XCC, 0XAA, 0X83, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X2C, 0XAA, 0X88, 0X66, 0X8A, 0X88, 0X65, 0X55, 0X63, 0X52,
    0X52, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X06, 0X63, 0X33, 0X33, 0X23, 0X8C, 0X55, 0X88,
    0X86, 0XCC, 0XA3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X8A, 0XAA, 0XCC, 0XFA, 0X00, 0X05, 0X68, 0XAA, 0XAA, 0XAA, 0X82, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X23, 0X56, 0X66, 0XAA, 0X86, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X05, 0X68,
    0X65, 0X8A, 0XCC, 0XA2, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0XAA, 0X6C, 0XC8, 0X00,
    0X00, 0X36, 0X86, 0XAC, 0XC0, 0X00, 0X00, 0X00, 0X00, 0X02, 0X23, 0X22, 0X23, 0X32, 0X20, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X28, 0XCA, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X63, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0XCA, 0XAA, 0X86, 0XAA, 0X88, 0X88, 0X66, 0X80, 0X00,
    0X28, 0X52, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X33, 0X33, 0X33, 0X36, 0X83, 0X56,
    0X58, 0XFC, 0X50, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0XAA, 0XAA, 0XCC, 0XFA, 0X00, 0X06, 0X88, 0XAC, 0XCC, 0XCA, 0XAA, 0X60, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X02, 0X36, 0X6A, 0XC8, 0X86, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X36,
    0X68, 0X56, 0XAA, 0XCC, 0XC6, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X5A, 0XA8, 0XCF, 0XC0,
    0X00, 0X26, 0X86, 0X8A, 0XC3, 0X00, 0X00, 0X00, 0X38, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X66,
    0X65, 0X53, 0X33, 0X22, 0X22, 0X22, 0X68, 0X8C, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA8, 0X53, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X0C, 0XCC, 0XCC, 0XCA, 0XAC, 0XAA, 0X86, 0X88, 0XA0, 0X00,
    0X00, 0X58, 0XA8, 0X30, 0X00, 0X00, 0X00, 0X00, 0X05, 0X8A, 0XAA, 0XAA, 0X88, 0X52, 0X65, 0X33,
    0X3C, 0XCC, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0XAA, 0XCA, 0XCC, 0XFA, 0X00, 0X06, 0X88, 0X8C, 0XCC, 0XCC, 0XCA, 0XA8, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X35, 0XAC, 0XA6, 0X56, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02,
    0X66, 0X88, 0X66, 0X8A, 0XCC, 0XCC, 0X83, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0XCA, 0X8C, 0XCF,
    0X30, 0X03, 0X68, 0X5A, 0XAC, 0X00, 0X00, 0X00, 0X88, 0X88, 0X8A, 0XAA, 0XAA, 0XAA, 0XAA, 0XA8,
    0X88, 0X88, 0X88, 0X66, 0X55, 0X8A, 0X88, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA8, 0X62, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0XCF, 0XCF, 0XCC, 0XAC, 0XCA, 0XAA, 0X88, 0XA5, 0X00,
    0X00, 0X00, 0X02, 0X58, 0X20, 0X00, 0X00, 0X02, 0X8A, 0XAC, 0XCC, 0XFF, 0XCC, 0XCA, 0X63, 0X33,
    0XCF, 0XC3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X8A, 0XCA, 0XCC, 0XCF, 0X00, 0X03, 0XAA, 0X8A, 0XCC, 0XCC, 0XCC, 0XCC, 0X20, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X3A, 0XCA, 0X85, 0X55, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X03, 0X66, 0X88, 0X65, 0X68, 0XAA, 0XCC, 0XCC, 0X86, 0X30, 0X00, 0X00, 0X00, 0X6C, 0XA8, 0XCC,
    0XFC, 0X00, 0X56, 0X66, 0XAC, 0X80, 0X00, 0X00, 0X8A, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA,
    0XAA, 0XAA, 0X88, 0X86, 0X6A, 0XAA, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X53, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0XCC, 0XA5, 0XCC, 0XCA, 0XA8, 0X8C, 0X50,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0XAC, 0XCF, 0XA3, 0X33, 0X56, 0XAA, 0X85, 0X8C,
    0XCA, 0X82, 0X22, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X6A, 0XCA, 0XCC, 0XCF, 0X20, 0X00, 0X8A, 0XAA, 0XAC, 0XCF, 0XCC, 0XCC, 0X50, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X8C, 0XAA, 0X83, 0X55, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X03, 0X66, 0X88, 0X86, 0X56, 0X8A, 0XAA, 0XAA, 0XAA, 0XAA, 0X86, 0X53, 0X28, 0XCC, 0X8A,
    0XCF, 0XFA, 0X23, 0X55, 0X6A, 0XAA, 0X00, 0X00, 0X68, 0XAA, 0XAA, 0XAA, 0XAA, 0XAC, 0XCC, 0XCC,
    0XCA, 0XAA, 0XA8, 0X88, 0XAC, 0XAC, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X52, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0XCC, 0XAC, 0XCA, 0XA8, 0XAA,
    0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X8A, 0X80, 0X05, 0X8A, 0XAA, 0XA8, 0XAC, 0X88,
    0X66, 0X88, 0XAA, 0XAA, 0XA8, 0X52, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X3C, 0XCA, 0XAC, 0XCF, 0XA0, 0X00, 0X2A, 0XAA, 0XAA, 0XCC, 0XCC, 0XCC, 0X30, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0XCC, 0XA8, 0X53, 0X65, 0X32, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X02, 0X56, 0X66, 0X88, 0X65, 0X55, 0X68, 0X88, 0X88, 0X88, 0X88, 0X88, 0X8A, 0XCA,
    0X8C, 0XCF, 0XFA, 0X63, 0X35, 0X8A, 0XA3, 0X00, 0X08, 0XAA, 0XAA, 0XAA, 0XAC, 0XCC, 0XCC, 0XCC,
    0XCC, 0XCA, 0XA8, 0X8A, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X33, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3F, 0XCC, 0XCC, 0XAA, 0X88,
    0XAA, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X8A, 0XCC, 0XA3, 0X00, 0X0F, 0XCA,
    0X86, 0X8A, 0X88, 0X88, 0XAA, 0XAA, 0XA8, 0X32, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X0C, 0XCC, 0XAC, 0XCC, 0XF0, 0X00, 0X00, 0X8A, 0XAA, 0XAA, 0XCC, 0XCA, 0X02, 0X35, 0X66,
    0X88, 0X88, 0X88, 0X88, 0X88, 0X86, 0X53, 0X6C, 0XCA, 0XA8, 0X35, 0X65, 0X32, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X02, 0X33, 0X35, 0X55, 0X53, 0X33, 0X35, 0X56, 0X68, 0X86, 0X66, 0X8C,
    0XC8, 0X8A, 0XCC, 0XFC, 0XA5, 0X36, 0X8A, 0XA5, 0X00, 0X88, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC,
    0XCC, 0XCC, 0XA8, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XAA, 0X50, 0X02, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0XCF, 0XCC, 0XCA, 0XAA,
    0XAA, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0XAA, 0XCA, 0X00, 0X00, 0X8C, 0XA8,
    0XC8, 0X35, 0XAC, 0XA8, 0X32, 0X58, 0XAA, 0XAA, 0XA6, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X06, 0XCC, 0XAC, 0XCC, 0XFC, 0X00, 0X00, 0X02, 0X6A, 0XAA, 0XAA, 0X88, 0XAA, 0XAC, 0XCC,
    0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XA8, 0X88, 0X55, 0XAA, 0XAA, 0XA8, 0X8A, 0XAC, 0XCC, 0XCF, 0XFF,
    0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFC, 0XCC, 0XCC, 0XCA, 0XAA, 0X88, 0X86, 0X53, 0X33, 0X33, 0X33,
    0X8A, 0XA8, 0X8A, 0XCC, 0XCC, 0XCA, 0X85, 0X58, 0X86, 0X35, 0X8A, 0XAC, 0XAA, 0XCC, 0XCC, 0XCC,
    0XCC, 0XCC, 0XCA, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA2, 0X00, 0X00, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3A, 0XCC, 0XCC,
    0XA2, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X28, 0XAC, 0X30, 0X2C, 0XFC, 0XA3,
    0XAC, 0XA3, 0X58, 0XCC, 0XA5, 0X00, 0X03, 0X8A, 0XCC, 0XCA, 0X52, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0XCC, 0XCA, 0XCC, 0XCF, 0X60, 0X00, 0X02, 0X23, 0X68, 0X88, 0XAA, 0XAA, 0XAA, 0XAA,
    0X88, 0X86, 0X66, 0X65, 0X55, 0X55, 0X56, 0X88, 0X8C, 0XCC, 0XCC, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF,
    0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCF, 0XFC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X86,
    0X55, 0X8A, 0XAA, 0X88, 0XAC, 0XCC, 0XCC, 0XCA, 0X86, 0X66, 0X66, 0X8A, 0XAA, 0XAC, 0XCC, 0XCC,
    0XCC, 0XCC, 0XCC, 0XCC, 0XCF, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X00, 0X00, 0X03, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X3A, 0XA8, 0X68, 0XAC, 0XFF, 0XCC, 0X85,
    0X3C, 0XC6, 0X56, 0X8C, 0XCA, 0X30, 0X00, 0X02, 0X6A, 0XCC, 0XCC, 0X83, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X5C, 0XCC, 0XAC, 0XCC, 0XF3, 0X25, 0X66, 0X88, 0X88, 0X88, 0X86, 0X33, 0X22, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X38, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC,
    0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAC, 0XCC, 0XCC, 0XAA, 0XAA, 0XAA,
    0X88, 0X86, 0X88, 0XAA, 0X86, 0X68, 0XAA, 0XCC, 0XCA, 0XA8, 0X86, 0X58, 0XAA, 0XAC, 0XCC, 0XCC,
    0XCC, 0XCC, 0XCC, 0XCF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA3, 0X00, 0X00, 0X03, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X8A, 0XCC, 0XFF, 0XFC, 0XCC, 0XC8, 0X53,
    0X3A, 0XC8, 0X66, 0X88, 0XCC, 0X80, 0X00, 0X00, 0X00, 0X3A, 0XCC, 0XCC, 0XA6, 0X20, 0X00, 0X00,
    0X00, 0X00, 0X08, 0XCC, 0XAA, 0XCC, 0XCF, 0X85, 0X65, 0X53, 0X32, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X36, 0XAA, 0XAA, 0XAC, 0XCC, 0XAA, 0XAA, 0XCC, 0XCC, 0XFF, 0XFF, 0XFF, 0XFF,
    0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFC, 0XCC, 0XCC, 0XCC, 0XAA, 0XAA, 0X88, 0X88,
    0X88, 0X88, 0X86, 0X68, 0X88, 0XA8, 0X86, 0X68, 0X8A, 0XA8, 0X88, 0X65, 0X58, 0X8C, 0XFC, 0XCC,
    0XCC, 0XCC, 0XCC, 0XCF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA0, 0X00, 0X00, 0X03, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X8A, 0XAC, 0XCC, 0XCC, 0XCA, 0X85, 0X53,
    0X3C, 0XC6, 0XA8, 0X88, 0XAC, 0XA3, 0X00, 0X00, 0X00, 0X00, 0X28, 0XCC, 0XCC, 0XAA, 0X83, 0X22,
    0X00, 0X00, 0X23, 0XCC, 0XCA, 0XAC, 0XCC, 0XF8, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X03, 0X88, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XCC, 0XCF, 0XFF, 0XFF, 0XFF, 0XFF, 0XCC, 0XCA,
    0XAC, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XCC, 0XCF, 0XFF, 0XFF, 0XFC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA,
    0XA8, 0X88, 0X86, 0X65, 0X55, 0X56, 0X88, 0X88, 0X86, 0X55, 0X55, 0X33, 0X33, 0X6C, 0XFC, 0XCC,
    0XCC, 0XCC, 0XCC, 0XCF, 0XFF, 0XFF, 0XFC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA0, 0X00, 0X00, 0X02, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X8A, 0XAA, 0XAA, 0X86, 0X53, 0X3A,
    0XFC, 0X80, 0X8C, 0XAA, 0X8A, 0XCA, 0X30, 0X00, 0X00, 0X00, 0X00, 0X03, 0X8A, 0XAC, 0XCC, 0XAA,
    0XAA, 0XAA, 0X88, 0X8C, 0XCC, 0XAA, 0XCC, 0XCF, 0X80, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X36, 0X88, 0X88, 0XAA, 0X88, 0X8A, 0XCC, 0XCC, 0XFF, 0XFF, 0XC8, 0X63, 0X03, 0X88, 0X88, 0X66,
    0X33, 0X33, 0X55, 0X53, 0X33, 0X33, 0X22, 0X00, 0X02, 0X33, 0X68, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC,
    0XCA, 0XAA, 0XAA, 0XAA, 0X88, 0X86, 0X65, 0X66, 0X66, 0X66, 0X66, 0X55, 0X33, 0X38, 0XCC, 0XCC,
    0XCC, 0XCC, 0XCC, 0XFF, 0XFF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XCA, 0XA3, 0X00, 0X00, 0X02, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X26, 0X53, 0X6C, 0XFC,
    0XC6, 0X00, 0X06, 0XCC, 0XCA, 0X8A, 0XAA, 0X52, 0X00, 0X00, 0X00, 0X00, 0X00, 0X35, 0X88, 0XAA,
    0XAA, 0XAA, 0X88, 0X53, 0XCC, 0XCA, 0XAC, 0XCC, 0XFC, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X36,
    0X68, 0X88, 0X88, 0X88, 0XAC, 0XCC, 0XCF, 0XFA, 0X83, 0X00, 0X00, 0X00, 0X00, 0X02, 0X58, 0X88,
    0X88, 0X86, 0X53, 0X33, 0X55, 0X66, 0X66, 0X66, 0X55, 0X55, 0X55, 0X33, 0X33, 0X35, 0X68, 0X8A,
    0XAC, 0XCC, 0XCC, 0XCC, 0XAA, 0XAA, 0XA8, 0X88, 0X86, 0X55, 0X55, 0X35, 0X55, 0X53, 0XAC, 0XCC,
    0XCC, 0XCC, 0XCC, 0XCF, 0XFF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XCA, 0XA8, 0X00, 0X00, 0X03, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X83, 0X5C, 0XFC, 0XA8,
    0X20, 0X00, 0X00, 0X03, 0X8A, 0XA8, 0X63, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X22, 0X20, 0X22, 0X23, 0X5A, 0XCC, 0XCA, 0XCC, 0XCF, 0XF8, 0X00, 0X00, 0X00, 0X02, 0X35, 0X66,
    0X88, 0X85, 0X8A, 0XAC, 0XCC, 0XCC, 0X62, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03,
    0X8A, 0X88, 0X88, 0X88, 0X86, 0X55, 0X55, 0X55, 0X55, 0X55, 0X66, 0X66, 0X66, 0X55, 0X55, 0X55,
    0X55, 0X55, 0X88, 0XAA, 0XCC, 0XCC, 0XCC, 0XCA, 0XAA, 0X88, 0X86, 0X55, 0X33, 0X33, 0X5C, 0XCC,
    0XCC, 0XCC, 0XCC, 0XCF, 0XFF, 0XFF, 0XFF, 0XFC, 0XCC, 0XCC, 0XCA, 0XAA, 0X80, 0X00, 0X00, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X05, 0X56, 0XFC, 0XA3, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X33, 0X35, 0X68, 0XAC, 0XCC, 0XAA, 0XCC, 0XCF, 0XF5, 0X00, 0X00, 0X35, 0X55, 0X68,
    0X55, 0X8A, 0XAA, 0XCC, 0XA3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X35, 0X8A, 0XAA, 0XAA, 0XAA, 0XAA, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88,
    0X88, 0X88, 0X86, 0X88, 0X8A, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0X88, 0X86, 0X55, 0X38, 0XCA,
    0XAC, 0XCC, 0XCC, 0XCC, 0XFF, 0XFF, 0XFF, 0XFC, 0XCC, 0XCC, 0XCC, 0XA8, 0XA8, 0X00, 0X03, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X06, 0X3C, 0XA6, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X05, 0X55, 0X68, 0X8A, 0XA8, 0XCC, 0XCC, 0XAC, 0XCC, 0XCF, 0XC5, 0X23, 0X35, 0X66, 0X53,
    0X8A, 0XAA, 0XCA, 0X50, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X25, 0X8A, 0XCC, 0XCC, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA,
    0XA8, 0X8A, 0XAA, 0XAA, 0XAA, 0X88, 0X65, 0X53, 0X35, 0X55, 0X55, 0X66, 0X65, 0X55, 0X53, 0X8A,
    0XAA, 0XCC, 0XCC, 0XCC, 0XFF, 0XFF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X83, 0X02, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X58, 0X80, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X25, 0X66, 0X68, 0XAC, 0XC2, 0X2C, 0XCC, 0XCA, 0XAC, 0XCC, 0XCF, 0XFA, 0X65, 0X33, 0X68,
    0XAA, 0XC8, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X35, 0X88, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X86, 0X68,
    0XAA, 0XCC, 0XAA, 0XA8, 0X86, 0X53, 0X58, 0X88, 0XA8, 0X88, 0X88, 0X66, 0X55, 0X53, 0X33, 0X38,
    0XAA, 0XCC, 0XCC, 0XCF, 0XFF, 0XFF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X66, 0X32, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X36, 0X80, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X36, 0X66, 0X88, 0XAC, 0XA2, 0X00, 0X8C, 0XCC, 0XCA, 0XAA, 0XCC, 0XCC, 0XFC, 0XAA, 0X88,
    0XA3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X25, 0X8C, 0XCC,
    0XCC, 0XAA, 0XA8, 0X65, 0X58, 0XAA, 0XCA, 0XAA, 0XAA, 0X88, 0X88, 0X88, 0X88, 0X88, 0X86, 0X55,
    0X8C, 0XFC, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XAA, 0XA8, 0X85, 0X33, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X02, 0X22, 0X22, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X26, 0X68, 0X88, 0XCC, 0X50, 0X00, 0X02, 0X8C, 0XCC, 0XCC, 0XAA, 0XAC, 0XCC, 0XCF, 0XFC,
    0X50, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X35, 0X8A, 0XCC, 0XCC, 0XCC, 0XAA,
    0XA8, 0X88, 0X66, 0XAC, 0XCC, 0XCC, 0XCC, 0XAA, 0XAA, 0XCC, 0XCC, 0XCC, 0XCA, 0X88, 0X65, 0X33,
    0X3F, 0XFF, 0XFF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X86, 0X32, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X26,
    0XAC, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XFF, 0XCC, 0X86, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X05, 0X66, 0X6A, 0XC6, 0X00, 0X00, 0X00, 0X03, 0XAC, 0XAA, 0XCC, 0XAA, 0XAA, 0XCC, 0XCC,
    0XCF, 0XFC, 0X85, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X58, 0XAC, 0XCF, 0XCC, 0XCC, 0XCC, 0XAA, 0XA8, 0X88,
    0X88, 0X88, 0XAC, 0XCC, 0XCC, 0XCC, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XA8, 0X65, 0X66, 0X66, 0X66,
    0X5F, 0XFF, 0XFF, 0XFF, 0XFC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X65, 0X33, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0XAF, 0XFF,
    0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCF, 0XFF, 0XFF, 0XFA, 0X62, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X56, 0X68, 0XA0, 0X00, 0X00, 0X00, 0X33, 0X35, 0X8A, 0XAA, 0XAA, 0XCA, 0XA8, 0XAA,
    0XCC, 0XCC, 0XCC, 0XFF, 0XCC, 0XA8, 0X53, 0X22, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X22, 0X35, 0X88, 0XAC, 0XCF, 0XFF, 0XCC, 0XCC, 0XCC, 0XCA, 0XAA, 0X88, 0X88, 0X88, 0X88,
    0X8A, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCF, 0XFC, 0XA8, 0X66, 0X66, 0X88, 0X8A, 0XA8, 0X53,
    0X2C, 0XFF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA8, 0X88, 0X65, 0X33, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0XCC, 0XCC, 0XCC,
    0XAA, 0XAA, 0XAA, 0XAA, 0X8A, 0XAA, 0XAA, 0XAC, 0XCC, 0XCC, 0XCF, 0XFF, 0XFA, 0X30, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X23, 0X50, 0X00, 0X00, 0X23, 0X33, 0X55, 0X33, 0X88, 0XAA, 0XAA, 0XAA, 0XAA,
    0XA8, 0X8A, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCF, 0XFC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCF, 0XFF,
    0XFF, 0XFF, 0XFC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X68,
    0XCC, 0XCC, 0XCC, 0XAC, 0XCC, 0XCC, 0XFF, 0XC5, 0X35, 0X68, 0X88, 0X8A, 0XAA, 0XC8, 0X32, 0X33,
    0X3A, 0XFF, 0XFF, 0XFF, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0XA8, 0X86, 0X53, 0X32, 0X30,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X8C, 0XAA, 0X88, 0X88,
    0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X8A, 0XCC, 0XCC, 0XCF, 0XFF, 0XC6, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X33, 0X55, 0X53, 0X68, 0X8A, 0X83, 0X6A, 0XAA, 0XAA,
    0XAA, 0XAA, 0XAA, 0X88, 0X88, 0X8A, 0XAA, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC,
    0XCC, 0XCC, 0XCC, 0XCA, 0XAA, 0XA8, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X63, 0X03, 0XAC,
    0XCC, 0XCC, 0XAC, 0XCC, 0XCF, 0XFC, 0X80, 0X05, 0X88, 0X88, 0X8A, 0XAC, 0XC8, 0X32, 0X33, 0X35,
    0X58, 0XFF, 0XCF, 0XFC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0X88, 0X88, 0X88, 0X53, 0X36,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X0A, 0X88, 0X86, 0X86, 0X65,
    0X53, 0X33, 0X32, 0X22, 0X23, 0X33, 0X33, 0X55, 0X66, 0X88, 0X88, 0X8A, 0XCC, 0XCC, 0XFF, 0XFC,
    0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X33, 0X55, 0X53, 0X58, 0X88, 0XA3, 0X00, 0X00, 0X03, 0X58,
    0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0XAA, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88,
    0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X86, 0X66, 0X65, 0X30, 0X00, 0X8C, 0XCC,
    0XCC, 0XAC, 0XCC, 0XCF, 0XFA, 0X30, 0X03, 0X88, 0X88, 0X8A, 0XCC, 0XCA, 0X32, 0X33, 0X33, 0X56,
    0X86, 0X8F, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA8, 0XAC, 0XCC, 0XA8, 0X88, 0X8C,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X86, 0X55, 0X33, 0X20, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X22, 0X33, 0X56, 0X88, 0X88, 0XAC, 0XCC, 0XCF,
    0XFF, 0XA2, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X33, 0X35, 0X53, 0X36, 0X88, 0XA8, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X23, 0X58, 0X8A, 0XAA, 0XAA, 0XAA, 0XAA, 0XA8, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88,
    0X88, 0X88, 0X88, 0X88, 0X88, 0X86, 0X63, 0X33, 0X35, 0X56, 0X65, 0X30, 0X00, 0X3A, 0XAA, 0XCC,
    0XAC, 0XCC, 0XCF, 0XFA, 0X00, 0X00, 0X68, 0X8A, 0X8A, 0XCC, 0XCC, 0X32, 0X23, 0X33, 0X36, 0X68,
    0X80, 0X0C, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0XAC, 0XCC, 0XCC, 0XAA, 0XC8, 0XAC,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X06, 0X53, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X33, 0X68, 0X88, 0X8A, 0XCC,
    0XCC, 0XFF, 0XF8, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X33, 0X35, 0X55, 0X35, 0X88, 0X8A, 0X50, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X02, 0X23, 0X33, 0X35, 0X55, 0X56, 0X55, 0X56, 0X66, 0X55, 0X55, 0X33,
    0X33, 0X33, 0X22, 0X20, 0X00, 0X00, 0X00, 0X35, 0X66, 0X66, 0X30, 0X00, 0X06, 0XAA, 0XAC, 0XAA,
    0XCC, 0XCC, 0XFA, 0X00, 0X00, 0X08, 0X88, 0X88, 0XAC, 0XCF, 0X80, 0X23, 0X33, 0X53, 0X68, 0X8A,
    0X20, 0X00, 0XCC, 0XAC, 0XCA, 0XAA, 0XCC, 0XCC, 0XCC, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0X8C,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X23, 0X36, 0X88, 0X88,
    0XAC, 0XCC, 0XCF, 0XFC, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X23, 0X33, 0X55, 0X35, 0X88, 0X8A, 0XA2, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X36, 0X88, 0X85, 0X20, 0X00, 0X00, 0X8A, 0XAA, 0XCA, 0XCC,
    0XCC, 0XFC, 0X20, 0X00, 0X00, 0X88, 0XA8, 0X8C, 0XCF, 0XC0, 0X00, 0X33, 0X35, 0X36, 0X88, 0XA5,
    0X00, 0X00, 0X2A, 0XAA, 0XAA, 0XAA, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCF, 0XC8, 0X6C,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X33, 0X68,
    0X88, 0X8A, 0XCC, 0XCF, 0XFF, 0XA2, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X23, 0X33, 0X55, 0X35, 0X88, 0X8A, 0XA6, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X02, 0X58, 0X88, 0X85, 0X20, 0X00, 0X00, 0X06, 0XAA, 0XAC, 0XAC, 0XCC,
    0XCF, 0X80, 0X00, 0X00, 0X06, 0X8A, 0X8A, 0XCC, 0XF8, 0X00, 0X02, 0X33, 0X55, 0X58, 0X88, 0XA0,
    0X00, 0X00, 0X00, 0X88, 0X88, 0X88, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XF8, 0X6A,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X23,
    0X56, 0X88, 0X88, 0XAC, 0XCC, 0XFF, 0XF8, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X02, 0X33, 0X35, 0X53, 0X35, 0X88, 0X8A, 0XAA, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X02, 0X22, 0X22, 0X22, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X02, 0X68, 0XAA, 0X83, 0X00, 0X00, 0X00, 0X00, 0X3A, 0XAA, 0XA8, 0XCC, 0XCC,
    0XF2, 0X00, 0X00, 0X00, 0X58, 0XA8, 0XAC, 0XCF, 0X20, 0X00, 0X03, 0X33, 0X53, 0X68, 0X8A, 0X30,
    0X00, 0X00, 0X00, 0X03, 0X66, 0X68, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XC8, 0X8C,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X02, 0X35, 0X68, 0X88, 0X8A, 0XCC, 0XCF, 0XFF, 0X82, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X23,
    0X33, 0X55, 0X33, 0X58, 0X88, 0X8A, 0XAA, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X23, 0X5A, 0XCC, 0XCC, 0XA8, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X03, 0X8A, 0XAA, 0X83, 0X00, 0X00, 0X00, 0X00, 0X00, 0X88, 0XAA, 0X8C, 0XCC, 0XFC,
    0X00, 0X00, 0X00, 0X02, 0X88, 0X88, 0XCC, 0XC0, 0X00, 0X00, 0X03, 0X33, 0X53, 0X88, 0X8A, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X25, 0X55, 0XAA, 0XCC, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XC8, 0X8C,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X23, 0X56, 0X88, 0X88, 0XAC, 0XCC, 0XFF, 0XFC, 0X83, 0X00, 0X22, 0X23, 0X33, 0X33,
    0X33, 0X55, 0X88, 0X88, 0XAA, 0XCC, 0X50, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X08, 0X88, 0XAC, 0XCC, 0XCC, 0XA5, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X03, 0X8A, 0XCA, 0XAA, 0X30, 0X00, 0X00, 0X00, 0X00, 0X05, 0X88, 0XA8, 0XAC, 0XCC, 0XC0,
    0X00, 0X00, 0X00, 0X05, 0X8A, 0X8C, 0XCC, 0X00, 0X00, 0X00, 0X03, 0X33, 0X53, 0X88, 0XA5, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X33, 0X8A, 0XAA, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XC8, 0X6C,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X23, 0X35, 0X68, 0X88, 0X8A, 0XAC, 0XCC, 0XFF, 0XFC, 0XCA, 0X86, 0X55, 0X56,
    0X68, 0X88, 0XAA, 0XAA, 0XA8, 0X30, 0X00, 0X00, 0X8C, 0XA0, 0X00, 0X00, 0X02, 0X68, 0X86, 0X53,
    0X00, 0X8A, 0XCC, 0XCC, 0XFC, 0XCC, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X03, 0XAC, 0XCC, 0X82, 0X8A, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0X8A, 0XA8, 0XCC, 0XCC, 0X00,
    0X00, 0X00, 0X00, 0X06, 0X88, 0XCC, 0XA0, 0X00, 0X00, 0X00, 0X23, 0X33, 0X35, 0X88, 0XA2, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X6A, 0X8A, 0XAA, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XC8, 0X5A,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X23, 0X56, 0X88, 0X88, 0X8A, 0XAA, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC,
    0XCA, 0XAA, 0X88, 0X63, 0X00, 0X00, 0X00, 0X0C, 0XAF, 0XA0, 0X00, 0X00, 0X8A, 0X88, 0X8A, 0X80,
    0X00, 0X03, 0X68, 0XAC, 0XCC, 0XCC, 0XA0, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X05,
    0XAC, 0XCC, 0X82, 0X00, 0XA8, 0X00, 0X00, 0X00, 0X00, 0X00, 0X28, 0X8A, 0X8A, 0XCC, 0XF0, 0X00,
    0X00, 0X00, 0X00, 0X28, 0X88, 0XCF, 0X20, 0X00, 0X00, 0X00, 0X22, 0X33, 0X35, 0X88, 0XA0, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0XA8, 0XAA, 0XAA, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X32, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X23, 0X35, 0X66, 0X88, 0X88, 0X88, 0X88, 0X88, 0X88,
    0X65, 0X33, 0X20, 0X00, 0X00, 0X00, 0X00, 0X8C, 0XAC, 0XCA, 0X20, 0X05, 0XA8, 0X86, 0XC8, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X38, 0X86, 0X53, 0X20, 0X00, 0X00, 0X00, 0X00, 0X02, 0X5A, 0XCC,
    0XCC, 0X82, 0X00, 0X00, 0XA8, 0X00, 0X00, 0X00, 0X00, 0X00, 0X38, 0X8A, 0X8C, 0XCC, 0X60, 0X00,
    0X00, 0X00, 0X00, 0X38, 0X6C, 0XCC, 0X30, 0X00, 0X00, 0X00, 0X02, 0X33, 0X35, 0X88, 0XA0, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X88, 0X88, 0X88, 0XAA, 0XAC, 0XCC, 0XCC, 0X8A, 0XAA,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X33, 0X33, 0X00, 0X00, 0X00, 0X00, 0X00, 0X55, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X22, 0X33, 0X33, 0X33, 0X32, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X6A, 0X88, 0XAF, 0XC8, 0X06, 0X88, 0X8A, 0XC5, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X05, 0XA8, 0X22, 0X33, 0X56, 0X8A, 0XCC, 0XCC, 0XCA,
    0X30, 0X00, 0X00, 0X00, 0X86, 0X00, 0X00, 0X00, 0X00, 0X00, 0X38, 0X88, 0X8A, 0XCC, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X38, 0X8C, 0XCA, 0X80, 0X00, 0X00, 0X00, 0X02, 0X33, 0X53, 0X88, 0XA0, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X58, 0X68, 0X88, 0X8A, 0XAA, 0XCC, 0XCC, 0X8A, 0XAA,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X3C, 0XCA, 0X86, 0X60, 0X00, 0X00, 0X00, 0X35, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0X88, 0X56, 0XCF, 0XC8, 0X68, 0X8A, 0XC6, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X23, 0X68, 0XAA, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0X63, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X53, 0X00, 0X00, 0X00, 0X00, 0X00, 0X36, 0X88, 0X8A, 0XC6, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X36, 0X8C, 0XCA, 0X82, 0X00, 0X00, 0X00, 0X02, 0X33, 0X53, 0X88, 0XA3, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X65, 0X66, 0X6A, 0XAA, 0XAA, 0XCC, 0X65, 0X8A,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X03, 0XCC, 0XA8, 0X66, 0X30, 0X00, 0X00, 0X05, 0X30, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X35, 0X33, 0X35, 0XCC, 0XA6, 0X88, 0XCA, 0X00,
    0X00, 0X00, 0X03, 0X6A, 0XCC, 0XCC, 0XCC, 0XCC, 0XCA, 0XA8, 0X86, 0X53, 0X20, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X68, 0X62, 0X00, 0X00, 0X00, 0X00, 0X06, 0X88, 0X8A, 0XC2, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X26, 0XAC, 0XAA, 0X86, 0X00, 0X00, 0X00, 0X02, 0X33, 0X53, 0X88, 0XA6, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X35, 0X33, 0X38, 0XAA, 0XAA, 0XAA, 0XA8, 0XA8,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03,
    0X32, 0X00, 0X00, 0XAF, 0XCC, 0XA8, 0X68, 0X00, 0X00, 0X00, 0X33, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X36, 0X68, 0X66, 0X53, 0X38, 0XC8, 0X68, 0XCA, 0X00,
    0X02, 0X5A, 0XCC, 0XCC, 0XA8, 0X53, 0X22, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X08, 0XAA, 0XA3, 0X00, 0X00, 0X00, 0X00, 0X05, 0X68, 0X8A, 0XC0, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X06, 0X8C, 0XA8, 0X88, 0X30, 0X00, 0X00, 0X02, 0X23, 0X53, 0X58, 0XAA, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X33, 0X35, 0X8A, 0XAA, 0XAA, 0X85, 0X68,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0XCC,
    0XCC, 0X80, 0X00, 0XCA, 0XCA, 0XA8, 0X66, 0X00, 0X00, 0X00, 0X30, 0X00, 0X00, 0X00, 0X00, 0X03,
    0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X36, 0X8A, 0XAA, 0XAA, 0XA8, 0X53, 0X6A, 0X68, 0XC8, 0X03,
    0XAC, 0XCA, 0X83, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X02, 0XAC, 0XCC, 0XC8, 0X00, 0X00, 0X00, 0X00, 0X00, 0X68, 0X68, 0XA0, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X03, 0X6C, 0XA8, 0X68, 0X63, 0X00, 0X00, 0X00, 0X23, 0X35, 0X38, 0X8A, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X33, 0X88, 0XAA, 0XAA, 0X65, 0X88,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X0C, 0XCA,
    0XA8, 0XA2, 0X00, 0X8A, 0XAA, 0XA8, 0X58, 0X00, 0X00, 0X05, 0XAC, 0X60, 0X00, 0X00, 0X25, 0X50,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X68, 0XAC, 0XCC, 0XCC, 0XCC, 0XA8, 0X55, 0X6A, 0XA5, 0X68,
    0X86, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X2A, 0XCC, 0XFF, 0XCA, 0X50, 0X00, 0X00, 0X00, 0X00, 0X28, 0X88, 0XA0, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X3A, 0XC8, 0X65, 0X66, 0X52, 0X00, 0X00, 0X22, 0X35, 0X36, 0X8A, 0X80,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X5A, 0X8A, 0XAA, 0X23, 0X38,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X0A, 0XCA,
    0XAA, 0X8A, 0X50, 0X88, 0X88, 0X88, 0X55, 0X60, 0X02, 0XCF, 0XA5, 0X33, 0X33, 0X55, 0X20, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X88, 0XAC, 0XA2, 0X00, 0X00, 0X03, 0X86, 0X66, 0X55, 0X52,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X5A, 0XCC, 0XFF, 0XCC, 0XA3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X68, 0X82, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X03, 0XA8, 0X86, 0X35, 0X66, 0X52, 0X00, 0X02, 0X33, 0X53, 0X88, 0XA2,
    0X00, 0X00, 0X00, 0X22, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0XA8, 0XAA, 0XA8, 0X88,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0XCA,
    0XAA, 0X88, 0XAA, 0X65, 0X68, 0X88, 0X65, 0X33, 0X3A, 0XFC, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X68, 0XAA, 0X00, 0X23, 0X33, 0X33, 0X38, 0X88, 0XAA, 0XCC,
    0X83, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X3A, 0XCC, 0XCC, 0XCC, 0XA6, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X83, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X3A, 0X88, 0X85, 0X35, 0X66, 0X65, 0X32, 0X23, 0X33, 0X36, 0X88,
    0X00, 0X23, 0X8C, 0X60, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X8A, 0X8A, 0XAA, 0X88,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X8C,
    0XAA, 0XAA, 0X88, 0XAA, 0XA8, 0X86, 0X65, 0X55, 0X5C, 0XFC, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X23, 0X50, 0X26, 0X8A, 0XAC, 0XC6, 0X38, 0XCC, 0XA8, 0X8A,
    0XCC, 0XC5, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X06, 0XAC, 0XCC, 0XCC, 0XA3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X6A, 0X88, 0X85, 0X55, 0X56, 0X68, 0X88, 0X88, 0X88, 0X8A,
    0XAC, 0XCF, 0X80, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X28, 0XA8, 0XA8, 0X88,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X6C,
    0XAA, 0XAA, 0XAA, 0XA8, 0X8A, 0XAA, 0XAA, 0XAA, 0X8C, 0XFF, 0XA0, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X68, 0XAC, 0XF8, 0X00, 0X08, 0X88, 0XC8, 0X56,
    0X6A, 0XCF, 0XC8, 0X33, 0X33, 0X82, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X5A, 0XC3, 0XCA, 0X60, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X38, 0XAA, 0X88, 0X88, 0X66, 0X66, 0X88, 0X8A, 0XAC,
    0XCC, 0X50, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X88, 0X88, 0X88,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X0A,
    0XCC, 0XCC, 0XCC, 0XCC, 0XAA, 0XAA, 0XA8, 0X88, 0X88, 0XCF, 0XA0, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X28, 0XAA, 0X20, 0X00, 0X8C, 0X85, 0XCC, 0X55,
    0X66, 0X68, 0XAC, 0XCC, 0XCA, 0XC2, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X05, 0XA2, 0XA6, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0X58, 0X8A, 0XAA, 0XCC, 0XCC, 0XAA, 0X86,
    0X68, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X05, 0X88, 0X88,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X28, 0XAA, 0XAC, 0XCC, 0XCC, 0XCC, 0XCC, 0XCC, 0XA5, 0XAF, 0XC0, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X03, 0XCC, 0XCC, 0XCF, 0XCA, 0X65, 0XAC, 0X36,
    0X88, 0X88, 0X8A, 0XAA, 0XAC, 0X80, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X22, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X33,
    0X33, 0X85, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X05, 0X88,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X55, 0X38, 0XAC, 0XCC, 0X88, 0XA8, 0X63, 0X35, 0XAC, 0X60, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X0A, 0XAA, 0XCC, 0XCC, 0XA8, 0X35, 0XAC, 0X20,
    0X38, 0XCC, 0XCC, 0XCC, 0XA3, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02,
    0X33, 0X33, 0X52, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0XA8, 0X66, 0X55, 0X53, 0X55, 0X35, 0X56, 0X68, 0XAC, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X02, 0X88, 0X88, 0X85, 0X22, 0X66, 0XAC, 0X60,
    0X00, 0X00, 0X22, 0X20, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X23, 0X23, 0X32, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X8C, 0XAA, 0XA8, 0X88, 0X88, 0X88, 0XAA, 0XA3, 0X3A, 0XA2, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0XAA, 0X8A, 0XC8,
    0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X02, 0X53, 0X6C, 0XAA, 0X8A, 0XAA, 0X32, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X08, 0XCA, 0XAA,
    0XA8, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X02, 0X68, 0X88, 0X30, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X20,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
    0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00, 0X00,
};

